<?php

/*
 *
 * File ini bagian dari:
 *
 * OpenSID
 *
 * Sistem informasi desa sumber terbuka untuk memajukan desa
 *
 * Aplikasi dan source code ini dirilis berdasarkan lisensi GPL V3
 *
 * Hak Cipta 2009 - 2015 Combine Resource Institution (http://lumbungkomunitas.net/)
 * Hak Cipta 2016 - 2025 Perkumpulan Desa Digital Terbuka (https://opendesa.id)
 *
 * Dengan ini diberikan izin, secara gratis, kepada siapa pun yang mendapatkan salinan
 * dari perangkat lunak ini dan file dokumentasi terkait ("Aplikasi Ini"), untuk diperlakukan
 * tanpa batasan, termasuk hak untuk menggunakan, menyalin, mengubah dan/atau mendistribusikan,
 * asal tunduk pada syarat berikut:
 *
 * Pemberitahuan hak cipta di atas dan pemberitahuan izin ini harus disertakan dalam
 * setiap salinan atau bagian penting Aplikasi Ini. Barang siapa yang menghapus atau menghilangkan
 * pemberitahuan ini melanggar ketentuan lisensi Aplikasi Ini.
 *
 * PERANGKAT LUNAK INI DISEDIAKAN "SEBAGAIMANA ADANYA", TANPA JAMINAN APA PUN, BAIK TERSURAT MAUPUN
 * TERSIRAT. PENULIS ATAU PEMEGANG HAK CIPTA SAMA SEKALI TIDAK BERTANGGUNG JAWAB ATAS KLAIM, KERUSAKAN ATAU
 * KEWAJIBAN APAPUN ATAS PENGGUNAAN ATAU LAINNYA TERKAIT APLIKASI INI.
 *
 * @package   OpenSID
 * @author    Tim Pengembang OpenDesa
 * @copyright Hak Cipta 2009 - 2015 Combine Resource Institution (http://lumbungkomunitas.net/)
 * @copyright Hak Cipta 2016 - 2025 Perkumpulan Desa Digital Terbuka (https://opendesa.id)
 * @license   http://www.gnu.org/licenses/gpl.html GPL V3
 * @link      https://github.com/OpenSID/OpenSID
 *
 */

use App\Enums\SistemEnum;
use App\Models\Artikel;
use App\Models\TeksBerjalan;
use Illuminate\Support\Facades\View;

defined('BASEPATH') || exit('No direct script access allowed');

class Teks_berjalan extends Admin_Controller
{
    public $modul_ini     = 'admin-web';
    public $sub_modul_ini = 'teks-berjalan';

    public function __construct()
    {
        parent::__construct();
        isCan('b');
    }

    public function index()
    {
        return view('admin.web.teks_berjalan.index');
    }

    public function tukar()
    {
        isCan('u');
        $data = $this->input->post('data');
        TeksBerjalan::setNewOrder($data);
        cache()->flush();

        return json(['status' => 1]);
    }

    public function datatables()
    {
        if ($this->input->is_ajax_request()) {
            $order = $this->input->get('order') ?? false;

            return datatables()->of(TeksBerjalan::with('artikel')->when(! $order, static fn ($q) => $q->orderBy('urut')))
                ->addColumn('drag-handle', static fn (): string => '<i class="fa fa-sort-alpha-desc"></i>')
                ->addColumn('ceklist', static function ($row) {
                    if (can('h')) {
                        return '<input type="checkbox" name="id_cb[]" value="' . $row->id . '"/>';
                    }
                })
                ->addIndexColumn()
                ->addColumn('aksi', static function ($row): string {
                    $aksi = '';

                    if (can('u')) {
                        $aksi .= '<a href="' . ci_route('teks_berjalan.form', $row->id) . '" class="btn btn-warning btn-sm"  title="Ubah Data"><i class="fa fa-edit"></i></a> ';

                        $aksi .= View::make('admin.layouts.components.tombol_aktifkan', [
                            'url'    => site_url("teks_berjalan/lock/{$row->id}"),
                            'active' => $row->status,
                        ])->render();
                    }

                    if (can('h')) {
                        $aksi .= '<a href="#" data-href="' . ci_route('teks_berjalan.delete', $row->id) . '" class="btn bg-maroon btn-sm"  title="Hapus Data" data-toggle="modal" data-target="#confirm-delete"><i class="fa fa-trash"></i></a> ';
                    }

                    return $aksi;
                })
                ->addColumn('teks', static function ($row): string {
                    $text = $row->teks;

                    $tautan = $row->tipe == 1 ? menu_slug('artikel/' . $row->tautan) : $row->tautan;

                    return $text . (' <a href="' . $tautan . '" target="_blank">' . $row->judul_tautan . '</a><br>');
                })
                ->addColumn('judul_tautan', static function ($row): string {
                    if ($row->tipe == 1) {
                        $tautan = menu_slug('artikel/' . $row->tautan);
                        $tampil = tgl_indo($row->artikel->tgl_upload) . ' <br> ' . $row->artikel->judul;
                    } else {
                        $tautan = $tampil = $row->tautan;
                    }

                    return '<a href="' . $tautan . '" target="_blank">' . $tampil . '</a>';
                })
                ->rawColumns(['drag-handle', 'ceklist', 'aksi', 'teks', 'judul_tautan'])
                ->orderColumn('teks', static function ($query, $order): void {
                    $query->orderBy('teks', $order);
                })
                ->make();
        }

        return show_404();
    }

    public function form($id = '')
    {
        isCan('u');
        $data['list_artikel'] = Artikel::where('tipe', 'statis')->limit(500)->orderBy('id', 'DESC')->get();
        if ($id) {
            $data['teks']        = TeksBerjalan::findOrFail($id);
            $data['form_action'] = ci_route('teks_berjalan.update', $id);
        } else {
            $data['teks']        = null;
            $data['form_action'] = ci_route('teks_berjalan.insert');
        }

        $data['daftar_tampil'] = SistemEnum::all();

        return view('admin.web.teks_berjalan.form', $data);
    }

    public function insert(): void
    {
        isCan('u');

        if (TeksBerjalan::create($this->validate($this->request))) {
            redirect_with('success', __('notification.created.success'));
        }

        redirect_with('error', __('notification.created.error'));
    }

    public function update($id = ''): void
    {
        isCan('u');
        if (TeksBerjalan::findOrFail($id)->update($this->validate($this->request, $id))) {
            redirect_with('success', __('notification.updated.success'));
        }
        redirect_with('error', __('notification.updated.error'));
    }

    public function delete($id = null): void
    {
        isCan('h');

        if (TeksBerjalan::destroy($this->request['id_cb'] ?? $id) !== 0) {
            redirect_with('success', __('notification.deleted.success'));
        }

        redirect_with('error', __('notification.deleted.error'));
    }

    public function lock($id = 0): void
    {
        isCan('u');
        if (TeksBerjalan::gantiStatus($id)) {
            redirect_with('success', __('notification.status.success'));
        }
        redirect_with('error', __('notification.status.error'));
    }

    protected function validate($request = [], $id = null)
    {
        $data = [
            'teks'         => htmlentities((string) $request['teks']),
            'tipe'         => (int) $request['tipe'], // 1 = 'Internal', 2 = 'Eksternal'
            'judul_tautan' => htmlentities((string) $request['judul_tautan']),
            'status'       => (int) $request['status'],
        ];

        $data['tautan'] = $request['tipe'] == '1' ? $request['tautan_internal'] : $request['tautan_eksternal'];

        if ($id === null) {
            $data['urut'] = TeksBerjalan::UrutMax();
        }

        return $data;
    }
}
